﻿// Version:		1.97

/****************************************************************\
HOW TO USE

#include "Map.as"

newMap(_root, "myMap", tileWidth, tileHeight);
myMap.loadMap("myMap.map");

var myXml:XML = myMap.saveMap();


Optional Low-level capabilities
newMap(_root, "myMap", tileWidth, tileHeight, [layers, enableSave, tileSize] );
newMap(_root, "myMap", tileWidth, tileHeight, 1, true, 32);

myMap.loadChipset("robotrek_indoors.png");
myMap.setTile(tileX,tileY, tileID);
myMap.setTile(tileX,tileY, tileID, [layerNum] );
myMap.setTile(3, 4, 96, 2 );

myMap.loadImage("map_image.jpg");
myMap.loadImage("map_image.jpg", [layerNum] );
myMap.loadImage("map_image.jpg", 2 );

\****************************************************************/



/*	Class Dependancies

arrayCoords.as

*/


/* List data members
	_this								References the "map" movieClip, which contains all the data & functions below.

	tileSize							How big the tiles are, in pixels.		(optional parameter, default = 16)
	layers								States the number of  tile / image  layers this map has.		(optional parameter, default = 1)
	tiles#_pic						The map's tiles, stored as a picture.  (There can be up to 9.)
	layer#_mc						Displays "tiles_pic"  OR  displays the map's image.  (There can be up to 9.)
	tile_array#						2D array that stores all the tile indexes for drawing & saving.
	chipset_pic					A picture of the chipset, which is the palette for drawing tiles.
	chipset_mc					Temporary MovieClip used to "catch" a newly loaded chipset, which is then copied to chipset_pic.
	tileWidth							How wide the map is, in tiles.
	tileHeight						How tall the map is, in tiles.
	chipset_loader				A MovieClipLoader used to load the chipset.
	chipsetIsLoading			Alters the behavior of setTile if a chipset is currently loading.
	backlog_array				Stores a list of accumulated functions to call after the chipset has finished loading.
	collision_array				2D array used to detect collisions.
	bitmapSupport				States whether this flash player support tiles.

	enableSave					Controls whether or not to record extra data for saving this as a map file.		(optional parameter, default = false)
	imageFile#						Stores the filename of the image.
	chipsetFile		`				Stores the filename of the chipset.
	oldChipsetFile				Stores the previous valid chipset filename.		(used to reset chipsetFile is chipset fails to load)
*/
	
/* List member functions
	onMapLoad										Custom function defined by external programs.  Called when a map finishes loading. 
	onChipsetLoad								Custom function defined by external programs.  Called when a chipset finishes loading. 
	onChipsetError								Custom function defined by external programs.  Called when a chipset FAILS to load.
	placeAtDepth									Returns a depth based on the number specified.  (1 = 10, 2 = 20)
	placeAboveDepth							Returns a depth just above the layer depth.  (1 = 11, 2 = 21)
	placeAboveAll								Returns a depth above all internal layers and movieclips.  (like getNextHighestDepth()
	getPixelCollision								Reads the collision of the specified pixel coordinates.
	getTileCollision								Reads the collision of the specified tile coordinates.
	setPixelCollision								Alters the collision of the specified pixel coordinates.
	setPixelCollision								Alters the collision of the specified tile coordinates.
	chipset_loader.onLoadError			Cancels "chipsetIsLoading" to return to normal operation.
	chipset_loader.onLoadInit			Copies a newly loaded chipset to chipset_pic.
	getCollide										Reads the collision value of a location.
	setCollide										Alters the collision at the specified location.
	setTile										Changes a tile at the specified location.		(array & image)
	drawTile											Reads and draws the tile at the specified location.		(image)
	loadChipset									Loads a new chipset, which is the palette for drawing tiles.
	loadImage										Loads an image to use as the map.  (functions and data remain intact)
	loadMap											Loads the specified map file & displays it.
	resetTileArray								Creates a new empty array to store tile image indexes  (erases the existing array)
	resetCollision									Creates a new empty attay to store collision data  (erases the existing array)
	makeLayer										Creates or resets data for the specified layer
	deleteLayer									Removes all data at specified layer
	drawAllTiles									Redraws all tiles on a specified layer
	
	
	saveMap											Returns XML data, which loadMap() can read from an XML file.  (enableSave must be "true")
	
	placeAtDepth									Returns the depth of a layer.  (1 = 10, 2 = 20, 3 = 30)
	placeAboveDepth							Returns the depth just above a layer.  (1 = 11, 2 = 21, 3 = 31)
	placeAboveAll								Returns a depth above all layers and movieClips inside the map;
	
	getAstarCost									This returns a collision as A* cost.
*/



function newMap( target, newName, depth, width, height, enableSave, tileSize, layers)
{
	trace("newMap()");
	// Create the map movieClip
	var depth = (depth) ? depth : 1;		// default depth = 1
	var _this = target.createEmptyMovieClip( newName, depth );
	
	
	
	// Initialize data members  / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / 
	var versionString = System.capabilities.version;
	var flashVersion = versionString.charAt(4);
	_this.bitmapSupport = (flashVersion >= 8) ? true : false;
	_this.enableSave = (enableSave) ? true : false;
	_this.tileSize = (tileSize == undefined) ? 16 : tileSize;		// shorthand for the IF statement	.			var  =  (condition) ? value if true : value if false
	_this.layers = (layers) ? layers : 1;		// undefined = 1
	_this.tileWidth = width;
	_this.tileHeight = height;
	_this.chipsetIsLoading = false;
	
	
	if (_this.enableSave)
	{
		_this.chipsetFile = "";
		_this.oldChipsetFile = "";
	}// if (enableSave)
	
	if(_this.bitmapSupport)
	{
		_this.chipset_loader = new MovieClipLoader();
		_this.backlog_array = new Array();
	}
	

	
	// Define Functions  / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / 
	
	
	
	// Create:  layer#_mc,   imageFile#,   tiles#_pic,   tile_array#
	_this.makeLayer = function( layerNum )
	{
		//trace("makeLayer("+layerNum+")");
		
		// Delete this layer if it already exists.
		_this.deleteLayer( layerNum );
		
		
		// Create layer#_mc		(layer1_mc)
		var layerName = "layer"+layerNum+"_mc";			// layer1_mc
		var thisLayer = _this.createEmptyMovieClip( layerName , _this.placeAtDepth(layerNum) );
		//trace("_this[layerName]: "+_this[layerName] );		// layer1_mc
		//trace("_this.layer1_mc: "+_this.layer1_mc );
		//_this[layerName] = thisLayer;			// redundant debugger helper
		
		// Create imageFile#		(imageFile1)
		_this["imageFile"+layerNum] = "";
		
		
		// Create tiles#_pic			(tiles1_pic)
		if (_this.bitmapSupport)
		{
			var imageName = "tiles"+layerNum+"_pic";		// tiles1_pic
			var thisImage = _this[imageName] = new flash.display.BitmapData(_this.tileWidth * _this.tileSize, _this.tileHeight * _this.tileSize, true, 0x00000000);
			thisLayer.attachBitmap( thisImage, thisLayer.getNextHighestDepth() );
			
			// Create tile_array#		(tile_array1)
			// // If this map will be saved, create a place to store tile image data
			_this.resetTileArray( layerNum );		// this function only exists when saving is enabled
		}// if (bitmapSupport)
		
		
		// Update layer count
		if (layerNum > _this.layers)
		{
			_this.layers = layerNum;
		}// (layerNum)
		
	}// makeLayer()
	
	
	
	_this.deleteLayer = function( layerNum )
	{
		// Delete layer#_mc		(layer1_mc)
		var layerName = "layer"+layerNum+"_mc";		// layer1_mc
		_this[layerName].removeMovieClip();		// erase existing layer at this depth
		
		
		// Delete tiles#_pic			(tiles1_pic)
		if (_this.bitmapSupport)
		{
			var tileName = "tiles"+layerNum+"_pic";		// tiles1_pic
			delete _this[tileName];							// erase existing image at this depth
		}// if (bitmapSupport)
		
		
		// Delete tile_array#		(tile_array1)
		var arrayName = "tile_array"+layerNum;		// tile_array1
		delete _this[arrayName];
		
		
		// Delete imageFile#		(imageFile1)
		var imageName = "imageFile"+layerNum;		//imageFile1
		delete _this[imageName];
		
		
		// If this is the last layer, reduce the known number of layers
		if (layerNum == _this.layers)
		{
			_this.layers--;
		}// (layerNum)
		
	}// deleteLayer()
		
	
	
	if(_this.bitmapSupport)
	{
		// Detect the loading of a new chipset
		_this.chipset_loader.onLoadError = function()
		{
			trace("chipset_loader.onLoadError()");
			// Chipset file not found
			_this.chipsetIsLoading = false;
			delete _this.backlog_array;
			_this.backlog_array = new Array();		// Ignore accumulated functions  (empty the array)

			// delete the unneeded container
			_this.chipset_mc.removeMovieClip();		// There's no image to catch
			
			// reset chipsetFile to previous value
			_this.chipsetFile = _this.oldChipset;
			
			// Call externally-defined function for error handling.
			_this.onChipsetError();
		}// onLoadError()
		
		
		
		_this.chipset_loader.onLoadInit = function()
		{
			trace("chipset_loader.onLoadInit()");
			
			// Clear the existing chipset image		(wipe the pixels, the image still exists)
			var destRect = new Rectangle(0, 0, 480, 256);
			_this.chipset_pic.fillRect( destRect, 0x00000000 );
			
			
			// Re-create the chipset bitmap  &  match the loaded image size
			_this.chipset_pic.dispose();
			delete _this.chipset_pic;
			var chipWidth = _this.chipset_mc._width;
			var chipHeight = _this.chipset_mc._height;
			_this.chipset_pic = new flash.display.BitmapData( chipWidth, chipHeight, true, 0xff000000 );
			
			
			// take a "snapshot" of the chipset & store it in its bitmap
			_this.chipset_pic.draw(_this.chipset_mc);
			
			
			// delete the unneeded container
			_this.chipset_mc.removeMovieClip();
			//trace("delete chipset_mc: "+_this.chipset_mc);		// confirm that it is gone
			
			
			// Make the background color transparent  (based on blank tile)
			// // read the color  (Blank pixel:	288 x 128)
			var maskColor = _this.chipset_pic.getPixel32(288, 128);
			var fullRect = new flash.geom.Rectangle(0, 0, _this.chipset_pic.width, _this.chipset_pic.height);
			var destPoint = new flash.geom.Point(0, 0);
			_this.chipset_pic.threshold(_this.chipset_pic, fullRect, destPoint, "==", maskColor, 0x00000000, 0x00FFFFFF, true);
			
			
			// Return to normal operation
			_this.chipsetIsLoading = false;
			
			
			// Do all the functions in the backlog
			_this.backlog_array.reverse();
			for (var index in _this.backlog_array)
			{
				// Read a stored function
				// // Since (for...in) reads from the last index up, we'll pop the last index.
				var thisFunction_obj:Object = _this.backlog_array.pop();
				
				// call the specified function & pass the expected arguments
				var callFunct:String = thisFunction_obj.funct;
				switch(callFunct)
				{
					case "drawTile":
						_this.drawTile(thisFunction_obj.tileX, thisFunction_obj.tileY, thisFunction_obj.layerNum);
					break;
				}// switch(callFunct)
				
			}// for...in (backlog_array)
			
			//trace("_this.chipset_pic: "+_this.chipset_pic);
			
			
			// Dispatch an event for external programs
			_this.onChipsetLoad();
			
		}//onLoadInit()
		
		
		
		// Alters the image
		_this.drawTile = function(tileX,tileY, layerNum)		// layerNum is optional
		{
			//trace("drawTile("+tileX+", "+tileY+", "+layerNum+")");
			layerNum = (layerNum) ? layerNum : 1;		// default layer = 1
			
			// create this layer if it doesn't exist.
			var imageName = "tiles"+layerNum+"_pic";
			if ( _this[imageName] == undefined)
			{
				_this.makeLayer( layerNum );
			}
			
			// read the id of this tile
			var thisTileArray = _this["tile_array"+layerNum];		// tile_array1
			var id = thisTileArray[tileX][tileY];
			
			// Draw this tile image
			if ( !_this.chipsetIsLoading )
			{
				// Chipset is ready
				//trace("\t copy & paste");
				#include "arrayCoords.as"
				// copy "id" tile from chipSet to "x, y" on map;
				// // figure out Copy & Paste coordinates
				var chipWidth = _this.chipset_pic.width / _this.tileSize;
				var output = ARRAY_2_COORDS(id, chipWidth);
				//trace("MAP.output.x: "+output.x+"   MAP.output.y: "+output.y+"   id: "+id);
				var copyArea = new flash.geom.Rectangle( output.x*_this.tileSize, output.y*_this.tileSize, _this.tileSize, _this.tileSize);
				var pastePoint = new flash.geom.Point(tileX*_this.tileSize, tileY*_this.tileSize);
				// // Do the Copy & Paste
				var thisTilePic = _this["tiles"+layerNum+"_pic"];		// tiles1_pic
				thisTilePic.copyPixels(_this.chipset_pic, copyArea, pastePoint);
				
			}else{
				
				// Chipset is loading
				//trace("\t add to backlog");
				// // Store this function in the backlog for later execution
				_this.backlog_array.push( {funct:"drawTile", tileX:tileX, tileY:tileY, layerNum:layerNum} );
			}// if (!chipsetIsLoading)
			
		}// drawTile()
		
		
		
		_this.setTile = function(tileX,tileY,id,layerNum)		// layerNum is optional
		{
			//trace("setTile("+tileX+", "+tileY+", "+id+", "+layerNum+")");
			layerNum = (layerNum) ? layerNum : 1;		// default layer = 1
			
			// create this layer if it doesn't exist.
			var imageName = "tiles"+layerNum+"_pic";		// tiles1_pic
			if ( _this[imageName] == undefined)
			{
				_this.makeLayer( layerNum );
			}
			
			// Change this tile in the tile_array
			var thisTileArray = _this["tile_array"+layerNum];		// tile_array1
			thisTileArray[tileX][tileY] = id;		// alter the array
			
			// Display the new tile
			_this.drawTile(tileX, tileY, layerNum);
			
		}// setTile()
		
		
	
		// drawAllTiles()
		if (_this.bitmapSupport)
		{
			//trace("drawAllTiles()");
			//trace("\t_this.tileWidth: "+_this.tileWidth);
			//trace("\t_this.tileHeight: "+_this.tileHeight);
			_this.drawAllTiles = function( layerNum )
			{
				layerNum = (layerNum) ? layerNum : 1;				// default = 1
				var thisArray = _this["tile_array"+layerNum];		// tile_array1
				if (thisArray)		// draw tiles only if their layer exists
				{
					// Re-create layer#_mc, in order to re-attach a new bitmap to it.		(layer#_mc is only PART of the layer)
					var layerName = "layer"+layerNum+"_mc";
					var thisLayer = _this[layerName];		// layer1_mc
					//thisLayer.removeMovieClip();
					thisLayer = _this.createEmptyMovieClip( layerName , _this.placeAtDepth(layerNum) );		// replaces the old movieClip  (via depth)
					//_this[layerName] = thisLayer;		// redundant debugger helper
					
					// Re-create tiles#_pic.		(tiles#_mc is only PART of the layer)
					var imageName = "tiles"+layerNum+"_pic";				// tiles1_pic
					delete _this[imageName];		// improves garbage-collection
					var thisImage = _this[imageName] = new flash.display.BitmapData(_this.tileWidth * _this.tileSize, _this.tileHeight * _this.tileSize, true, 0x00000000);
					
					thisLayer.attachBitmap( thisImage, thisLayer.getNextHighestDepth() );		// makes  "layer#_mc"  display  "tiles#_pic"
					
					// display all the tiles
					for (var y = 0; y < _this.tileHeight; y++)
					{
						for (var x = 0; x < _this.tileWidth; x++)
						{
							// Draw this tile
							_this.drawTile(x,y, layerNum );		// draws the tile image
						}// for x
					}// for y
					
				}// if (thisArray)
			}// drawAllTiles()
		}// if (enableSave) 
		
		
		
		// resetTileArray()
		// // Create tile_array
		_this.resetTileArray = function( layerNum )		// layerNum is optional
		{
			//trace("resetTileArray()");
			layerNum = (layerNum) ? layerNum : 1;
			
			var thisArrayName = "tile_array"+layerNum;		// tile_array1
			var thisTileArray = _this[thisArrayName];
			delete thisTileArray;
			
			_this[thisArrayName] = new Array(_this.tileWidth);
			var thisTileArray = _this[thisArrayName];
			for (var x = 0; x < _this.tileWidth; x++)
			{
				thisTileArray[x] = new Array(_this.tileHeight);
				for (var y = 0; y < _this.tileHeight; y++)
				{
					thisTileArray[x][y] = 0;
				}// for y
			}// for x
		}// resetTileArray()
		
		
		
		_this.loadChipset = function(filename)
		{
			//trace("loadChipset("+filename+")");
			_this.chipsetIsLoading = true;				// Postpone drawTile() functions.
			// Create a movieclip to "catch" the loaded image
			_this.createEmptyMovieClip("chipset_mc", 1000 );
			
			// The code below might be unneccessary
			if (_this.chipset_pic == undefined)
			{
				_this.chipset_pic = new flash.display.BitmapData( 480, 256, true, 0xff000000 );
			}
			// End remove
			
			// Load the image into that movieClip
			_this.chipset_loader.loadClip(filename, _this.chipset_mc);
			
			_this.oldChipset = _this.chipsetFile;
			if (_this.enableSave)
			{
				_this.chipsetFile = filename;			
			}// if (enableSave)
		}// loadChipset()
		
	}// if (bitmapSupport)
	
	
	
	// Collision functions
	_this.getPixelCollision = function(x,y)
	{
		var tileX = Math.floor(x / _this.tileSize);
		var tileY = Math.floor(y / _this.tileSize);
		return _this.collision_array[tileX][tileY];
	}// getPixelCollision()
	
	
	
	_this.setPixelCollision = function(x,y,newValue)
	{
		var tileX = Math.floor(x / _this.tileSize);
		var tileY = Math.floor(y / _this.tileSize);
		_this.collision_array[tileX][tileY] = newValue;
	}// setPixelCollision()
	
	
	
	_this.getTileCollision = function(x,y)
	{
		return _this.collision_array[x][y];
	}// getTileCollision()
	
	
	
	_this.setTileCollision = function(x,y,newValue)
	{
		_this.collision_array[x][y] = newValue;
	}// setTileCollision()
	
	
	
	// Create collision array
	_this.resetCollision = function()
	{
		//trace("resetCollision()");
		_this.collision_array = new Array(_this.tileWidth);
		for (var x = 0; x < _this.tileWidth; x++)
		{
			_this.collision_array[x] = new Array(_this.tileHeight);
			for (var y = 0; y < _this.tileHeight; y++)
			{
				_this.collision_array[x][y] = 0;
			}// for y
		}// for x
	}// resetCollision()
	
	
	
	_this.loadImage = function(filename, layerNum)
	{
		//trace("loadImage()");
		layerNum = (layerNum) ? layerNum : 1;			// default layer = 1
		var layerName = "layer"+layerNum+"_mc";		// thisLayer = _this[layer1_mc]
		var thisLayer = _this[layerName];		// layer1_mc
		if ( thisLayer == undefined)
		{
			_this.makeLayer( layerNum );
		}else{
			if (_this.enableSave)
			{
				_this.resetTileArray( layerNum );
				_this["imageFile"+layerNum] = filename;		// imageFile1
				//_this.imageFile = filename;						// ??
			}// if (enableSave)
		}// if (thisLayer == undefined)
		
		var thisLayer = _this["layer"+layerNum+"_mc"];		// layer1_mc
		loadMovie( filename, thisLayer);

	}// loadImage()
	
	
	
	_this.loadMap = function(mapFile)
	{
		//trace("loadMap( "+mapFile+" )");
		// Load the .map file
		XML.prototype.ignoreWhite = true;
		var newMap_xml:XML = new XML();
		newMap_xml.onLoad = function()
		{
			// Delete all layers & images
			//trace("\tnewMap_xml.onLoad()");
			//trace(newMap_xml);
			for (var i = 1; i <= _this.layers; i++)
			{
				_this.deleteLayer(i);
			}
	
			// // Store the loaded data
			_this.layers = Number(newMap_xml.firstChild.attributes.layers);
			_this.tileWidth = Number(newMap_xml.firstChild.attributes.tileWidth);
			_this.tileHeight = Number(newMap_xml.firstChild.attributes.tileHeight);
			_this.tileSize = Number(newMap_xml.firstChild.attributes.tileSize);
			
			// // // Store collision
			var readString:String = newMap_xml.firstChild.attributes.collision_array;
			var readIndex:Number = 0;
			var nextBit = function()
			{
				var readData = Number( readString.charAt(readIndex) );
				readIndex++;		// Get ready to read the next value, next time
				return readData;
			}
			_this.resetCollision();		// erase & re-create the collision array
			for (var y = 0; y < _this.tileHeight; y++)
			{
				for (var x = 0; x < _this.tileWidth; x++)
				{
					_this.collision_array[x][y] = nextBit();		// read the next value;
				}// for x
			}// for y
			
			var usesTiles = false;						// Whether or not to load a chipset.		(if these's even one layer of tile data, then a chipset is needed)
			_this.chipsetIsLoading = true;		// Postpone drawTile() functions until the chipset is read. (which happens later)
			
			// Load the rest of the data, one layer at a time
			for (var layerNum = 1; layerNum <= _this.layers; layerNum++)
			{
				// Create this layer
				_this.makeLayer( layerNum );
				
				// determine what type of layer this is
				var tileString = newMap_xml.firstChild.attributes["tile_array"+layerNum];		// tile_array1
				var imageFilename = newMap_xml.firstChild.attributes["imageFile"+layerNum];		// imageFile1
				var layerType = (tileString) ? "tile" : "image";
				
				
				
				if ( layerType == "tile" )
				{
					// This is a tile-based layer
					usesTiles = true;		// load the chipset after this
					
					if(_this.bitmapSupport)
					{
						// // Load & Draw all the tiles
						_this.resetTileArray( layerNum );
						//var tilePicName = "tiles"+layerNum+"_pic";		// tiles1_pic
						var readString:String = newMap_xml.firstChild.attributes["tile_array"+layerNum];
						var readIndex:Number = 0;	
						var nextTile = function()
						{
							var readData = readString.substr( readIndex, 2 );
							readIndex += 2;		// Values are read 2 at a time
							if (readData != "0x")
							{
								var output:Number = parseInt( readData, 36 );	// Convert base36 -> Number
							}else{
								var output:Number = 33;
							}
							return output;
						}
						for (var y = 0; y < _this.tileWidth; y++)
						{
							for (var x = 0; x < _this.tileHeight; x++)
							{
								var tileValue = nextTile();
								// Draw this tile
								_this.setTile(x,y, tileValue, layerNum );
								
								// Record this tile for later saving
								var thisArray = _this["tile_array"+layerNum];		// tile_array1
								thisArray[x][y] = tileValue;
								
							}// for x
						}// for y
						
					}// if (bitmapSupport)
					
				}else{
					// This is an image-based layer
					
					// // Load the specified image
					var attrName = "imageFile"+layerNum;		// imageFile1
					var imageFile = newMap_xml.firstChild.attributes[attrName];
					_this.loadImage(imageFile, layerNum);
				} // if (layerType == "tile")  else...
			}// for (layerNum)
			
			
			// Load the specified chipset
			if ( usesTiles )
			{
				if(_this.bitmapSupport)
				{
					var newChipset = newMap_xml.firstChild.attributes.chipsetFile;
					_this.loadChipset(newChipset);	
				}
			}// if (usesTiles)
			
			
			// Dispatch an event for external programs to define
			_this.onMapLoad();
			
		}// onLoad()
		//trace("Loading map file.  onLoad() should occur soon.");
		newMap_xml.load( mapFile );
		
	}// loadMap()
	
	
	
	if (_this.enableSave)
	{
		_this.saveMap = function()
		{
			//trace("saveMap()");
			var newMap_xml:XML = new XML();
			var baseNode:XMLNode = newMap_xml.createElement("map");
			newMap_xml.appendChild(baseNode);
			var thisXml = newMap_xml.firstChild;		// convenient reference
			//thisXml.attributes[newVariable] = newValue;
			
			
			// Write the data
			thisXml.attributes["layers"] = _this.layers;
			thisXml.attributes["tileSize"] = _this.tileSize;
			thisXml.attributes["tileWidth"] =  _this.tileWidth;
			thisXml.attributes["tileHeight"] =  _this.tileHeight;
			
			
			// write collision array
			var collisionString = "";
			for (var y = 0; y< _this.tileHeight; y++)
			{
				for (var x = 0; x < _this.tileWidth; x++)
				{
					collisionString += _this.collision_array[x][y];
				}// x
			}// y
			thisXml.attributes["collision_array"] =  collisionString;
			
			
			var hasChipset = false;		// states whether there is a chipset  (based on what's below)
			
			
			// write tile_array#  /  imageFile#
			for (var layerNum = 1; layerNum <= _this.layers; layerNum++)
			{
				// Detect layer type
				if (_this["imageFile"+layerNum].length > 0 )				// imageFile1
				{
					// This layer uses an image
					var attrName = "imageFile"+layerNum;		// imageFile1
					thisXml.attributes[attrName] =  _this[attrName];
				
				}
				else if (_this["tile_array"+layerNum].length > 0 )		// tile_array1
				{
					// This layer uses tiles
					hasChipset = true;		// Tiles use the chipset, so save the chipset later
					
					var thisTileArray = _this["tile_array"+layerNum];		// tile_array1
					var tileString:String = "";
					for (var y = 0; y < _this.tileHeight; y++)
					{
						for (var x = 0; x < _this.tileWidth; x++)
						{
							var readTile:Number = thisTileArray[x][y];
							var readTileString = readTile.toString(36);		// Convert Number -> base36
							if (readTileString.length == 1)
							{
								readTileString = "0"+readTileString;
							}
							tileString += readTileString;
						}// x
					}// y
					var attrName = "tile_array"+layerNum;		// tile_array1
					thisXml.attributes[attrName] =  tileString;
					
				}// if (imageFile1  else  tile_array1)
			}// for (layers)
			
			
			// write the chipset
			if (hasChipset)
			{
				thisXml.attributes["chipsetFile"] =  _this.chipsetFile;
			}// if (hasChipset)
			
			
			// - - - - - - - - - - - - - - - - - - - - - - - - 
			return newMap_xml;
			
		}// saveMap()
	}// if (enableSave)
	
	
	
	_this.scroll = function( pixelX, pixelY, screenWidth, screenHeight)
	{
		var halfWidth = screenWidth/2;
		var mapWidth = _this.tileWidth*_this.tileSize;
		if (pixelX < halfWidth)
		{
			// scroll to left edge
			_this._x = 0;
		}
		else if (pixelX > mapWidth-halfWidth )
		{
			// scroll to right edge
			_this._x = 0 - mapWidth + screenWidth;
		}else{
			// scroll freely, centering on pixelX
			_this._x = 0 - pixelX + (screenWidth/2);
		}// if (pixelX)
		
		
		var halfHeight = screenHeight/2;
		var mapHeight = _this.tileHeight*_this.tileSize;
		if (pixelY < halfHeight)
		{
			// scroll to top edge
			_this._y = 0;
		}
		else if (pixelY > mapHeight-halfHeight )
		{
			// scroll to bottom edge
			_this._y = 0 - mapHeight + screenHeight;
		}else{
			// scroll freely, centering on pixelX
			_this._y = 0 - pixelY + (screenHeight/2);
		}// if (pixelY)
		
	}// scroll()
	
	
	
	// Internal depth functions
	_this.placeAtDepth = function( layerNum )
	{
		return	layerNum * 10;
	}// placeAtDepth()
	
	
	
	_this.placeAboveDepth = function( layerNum )
	{
		// placed above internal layer#_mc
		return	layerNum * 10 + 1;
	}// placeAboveDepth()
	
	
	
	_this.placeAboveAll = function()
	{
		// placed above ALL internal layer#_mc
		var nextDepth = _this.getNextHighestDepth();
		var altDepth = _this.layers * 10 + 1;
		var output = (nextDepth>altDepth) ? nextDepth : altDepth;		// use whichever one is higher
		//trace("nextDepth: "+nextDepth);
		//trace("altDepth: "+altDepth);
		return output;
	}// placeAboveAll()
	
	
	
	_this.getAstarCost = function(x,y)
	{
		var cost = 1;
		if (_this.collision_array[x][y] > 0)
		{
			cost = Infinity;
		}// if collision
		// - - - - - - - - - - - - 
		return cost;
	}// getAstarCost()
	
	
	
	// Setup Objects  / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / 
	// Create collision_array
	_this.resetCollision();
	//trace("_this.resetCollision: "+_this.resetCollision);

	// Create Layers
	// // Create layer#_mc		(layer1_mc)
	// // Create tiles#_pic			(tiles1_pic)
	// // Create tile_array#		(tile_array1)
	// // Create imageFile#		(imageFile1)
	for (var layerNum = 1; layerNum <= _this.layers; layerNum++)
	{
		//trace("_this.makeLayer: "+_this.makeLayer);
		_this.makeLayer( layerNum );
	}

	
	
	// / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / / 
	// return a reference to this map
	return _this;
	
}// newMap()
